import { formatMinutes, isMobile, numberFormatter } from "./common.js";

window.Charts = class Charts {
  static initPieChart(inputData, id, title = "", type = "") {
    var data = [];
    var labels = [];
    var others = 0;
    var isEmpty = !inputData.length || !inputData[0].value;

    for (var i in inputData) {
      var single = inputData[i];
      if (single.value) {
        isEmpty = false;
      }
      if (i > 98) {
        others += single["value"];
      } else {
        labels.push(single["id"]);
        data.push(single["value"]);
      }
    }

    if (isEmpty) {
      data.push(1);
      labels.push("No data");
    }

    if (others > 0) {
      data.push(others);
      labels.push("Other");
    }

    var backgroundColor = Object.values(HISTOGRAM_COLORS).map(color => {
      return hexToRgb(color, 0.7);
    });

    // Add a different color at the end if the colors of the first and last bars are the same
    if (data.length % backgroundColor.length == 1) {
      backgroundColor.push(hexToRgb(Object.values(HISTOGRAM_COLORS)[1], 0.7));
    }
    var dataOptions = {
      labels: labels,
      datasets: [
        {
          data: data,
          backgroundColor: backgroundColor,
          hitRadius: 20,
          borderWidth: 0,
          hoverOffset: 10
        }
      ]
    };
    var tooltipLabel = "";
    var metric = "";
    if (type == "reading-time") {
      tooltipLabel = "Reading time: ";
      metric = "min.";
    } else if (type == "websites" || type == "tabs-count") {
      tooltipLabel = "Number of tabs: ";
    } else if (type == "websites-count") {
      tooltipLabel = "Number of websites: ";
    }
    var chartOptions = {
      plugins: {
        legend: {
          display: false
        },
        tooltip: {
          enabled: false,
          xAlign: "right",
          yAlign: "top",
          callbacks: {
            label: function(tooltipItem) {
              return type == "reading-time"
                ? tooltipLabel + formatMinutes(tooltipItem.parsed)
                : tooltipLabel + numberFormatter(tooltipItem.parsed) + (metric ? ` ${metric}` : "");
            }
          },
          external: Charts.tooltipHandler
        }
      },
      responsive: true,
      maintainAspectRatio: false,
      aspectRatio: 1,
      layout: {
        autoPadding: false,
        padding: 20
      },
      cutout: "70%"
    };

    if (isEmpty) {
      chartOptions.plugins.tooltip.enabled = false;
    }

    var chartElement = $(`#${id}`);
    if (title) {
      chartElement
        .closest(".chart-wrapper")
        .find(".chart-title")
        .text(title);
    }
    var chart = new Chart(chartElement, {
      type: "doughnut",
      data: dataOptions,
      options: chartOptions
    });
    return chart;
  }

  static initAreaChart(inputData, id, title) {
    var data = [];
    var labels = [];

    for (var i in inputData) {
      var single = inputData[i];
      labels.push(single["id"]);
      data.push(single["value"]);
      if (i > 48) {
        break;
      }
    }

    var chartElement = $(`#${id}`);
    var colors = Object.values(HISTOGRAM_COLORS).map(color => {
      return hexToRgb(color, 0.7);
    });
    // Add a different color at the end if the colors of the first and last bars are the same
    if (data.length % colors.length == 1) {
      colors.push(hexToRgb(Object.values(HISTOGRAM_COLORS)[1], 0.7));
    }
    var data = {
      labels: labels,
      datasets: [
        {
          data: data,
          hitRadius: 10,
          borderWidth: 1,
          backgroundColor: colors
        }
      ]
    };
    chartElement
      .closest(".chart-wrapper")
      .find(".chart-title")
      .text(title);
    var chart = new Chart(chartElement, {
      type: "polarArea",
      data: data,
      options: {
        interaction: {
          mode: "point",
          intersect: false
        },
        backdropColor: "blue",
        scales: {
          r: {
            max: 100,
            beginAtZero: true,
            grid: {
              z: 1
            },
            ticks: {
              callback: function(value) {
                return value + "%";
              },
              stepSize: 25,
              backdropColor: "transparent",
              color: "black",
              z: 1
            }
          }
        },
        plugins: {
          legend: {
            display: false
          },
          tooltip: {
            enabled: false,
            xAlign: "right",
            yAlign: "top",
            callbacks: {
              label: function(tooltipItem) {
                return "Overlap with main topic: " + tooltipItem.raw + "%";
              }
            },
            external: Charts.tooltipHandler
          }
        },
        layout: {
          autoPadding: false,
          padding: {
            left: 20,
            right: 20,
            bottom: 0,
            top: 0
          }
        },
        responsive: true,
        aspectRatio: 1,
        maintainAspectRatio: false
      }
    });
    return chart;
  }

  static getTooltip(chart) {
    let tooltipElement = $(chart.canvas).siblings(".custom-tooltip");

    if (!tooltipElement.length) {
      tooltipElement = $(`
        <div class="custom-tooltip hidden">
          <span class="tooltip-title"></span>
          <div class="tooltip-body">
            <span class="label-color"></span>
            <span class="label"></span>
          </div>
        </div>
      `);
      $(chart.canvas)
        .parent()
        .append(tooltipElement);
    }

    return tooltipElement;
  }

  static tooltipHandler(context) {
    if (isMobile()) {
      // Prevent fast clicks when the tooltip moves to its new location
      setTimeout(() => {
        updateTooltip(context);
      }, 100);
    } else {
      updateTooltip(context);
    }

    function updateTooltip(context) {
      var { chart, tooltip } = context;
      var tooltipElement = Charts.getTooltip(chart);

      if (tooltip.opacity === 0) {
        $(".custom-tooltip").css("opacity", 0);
        tooltipElement.css("display", "none");
        return;
      }

      if (tooltip.body) {
        var titleLines = tooltip.title || [];
        var bodyLines = tooltip.body.map(b => b.lines);

        var isEmpty = titleLines[0] === "No data";
        if (!isEmpty) {
          $(tooltipElement)
            .find(".label-color")
            .css("background", tooltip.labelColors[0].backgroundColor);
        } else {
          $(tooltipElement)
            .find(".label-color")
            .remove();
        }
        $(tooltipElement)
          .find(".tooltip-title")
          .text(titleLines[0]);
        $(tooltipElement)
          .find(".label")
          .text(isEmpty ? "" : bodyLines[0]);
      }

      var { offsetLeft: positionX, offsetTop: positionY } = chart.canvas;
      $(tooltipElement).removeClass("hidden");
      var areaWidth = tooltip.dataPoints[0].element.outerRadius - tooltip.dataPoints[0].element.innerRadius + 5;
      $(tooltipElement).css({
        opacity: 1,
        display: "block",
        left:
          positionX +
          tooltip.caretX +
          (tooltip.caretX > tooltip.dataPoints[0].element.innerRadius
            ? areaWidth
            : tooltip.dataPoints[0].element.innerRadius),
        top: positionY + tooltip.caretY + areaWidth
      });
    }
  }

  static updateChart(data, id) {
    var chartToUpdate = $(`#${id}`)[0];
    for (var chart of Object.values(Chart.instances)) {
      if (chart.canvas == chartToUpdate) {
        chart.data.labels = data.map(single => {
          return single["id"];
        });
        chart.data.datasets[0].data = data.map(single => {
          return single["value"];
        });
        chart.update();
        break;
      }
    }
  }
};

function generateCustomLegend(chart, chartId, chartType = "pie") {
  var legendContainer = document.getElementById(chartId).nextElementSibling;
  var data = chart.data;
  var labels = data.labels;
  if (chartType == "bubble") {
    for (var dataset of data.datasets) {
      labels.push(dataset.label[0]);
    }
  }
  legendContainer.innerHTML = "";
  legendContainer.id = "chart-" + chart.id;

  for (var i = 0; i < labels.length; i++) {
    if (chartType == "pie" && !data.datasets[0].data[i]) {
      continue;
    }
    var legendItem = document.createElement("div");
    var legendItemColor = document.createElement("span");
    var legendItemText = document.createElement("span");

    legendItem.id = legendContainer.id + "-" + i;
    legendItem.classList.add("legend-item");
    var backgroundColor = chartType == "pie" ? data.datasets[0].backgroundColor[i] : data.datasets[i].backgroundColor;
    if (chartType == "bubble") {
      backgroundColor = backgroundColor.split(",", 3).join(",") + ", 1)";
    }
    legendItemColor.style.backgroundColor = backgroundColor;
    legendItemColor.classList.add("legend-color");
    legendItemText.textContent = labels[i] ? labels[i] : "";
    legendItem.checked = true;

    legendItem.addEventListener("click", function() {
      var datasetIndex = parseInt(this.id.split("-").pop());
      this.checked = !this.checked;
      if (chartType == "bubble") {
        chart.getDatasetMeta(datasetIndex).data[0].hidden = !this.checked;
      } else {
        chart.getDatasetMeta(0).data[datasetIndex].hidden = !this.checked;
      }
      if (this.checked) {
        this.classList.remove("disabled");
      } else {
        this.classList.add("disabled");
      }
      chart.update();
    });

    legendItem.appendChild(legendItemColor);
    legendItem.appendChild(legendItemText);
    legendContainer.appendChild(legendItem);
  }
}

function hexToRgb(hex, opacity = 1) {
  hex = hex.replace(/^#/, "");
  const r = parseInt(hex.substring(0, 2), 16);
  const g = parseInt(hex.substring(2, 4), 16);
  const b = parseInt(hex.substring(4, 6), 16);

  return `rgba(${r}, ${g}, ${b}, ${opacity})`;
}
