import { POPUP_MESSAGES, VALIDATION_MESSAGES, API_VERSION, API_URL, HEADERS } from "./config.js";

export function addPopupMessage(type, args = {}) {
  var messageText = POPUP_MESSAGES[type];
  if (!messageText) {
    return;
  }
  if (messageText.indexOf("%%GROUP%%") > -1) {
    messageText = messageText.replace("%%GROUP%%", args.group);
  }
  if (messageText.indexOf("%%FIELD%%") > -1) {
    messageText = messageText.replace("%%FIELD%%", args.field);
  }
  if (messageText.indexOf("%%TABS_OR_TAB%%") > -1) {
    messageText = messageText.replace("%%TABS_OR_TAB%%", args.tabsCount > 1 ? "Tabs" : "Tab");
  }
  if (!$(".pop-up-message").length) {
    $("body").append(`
      <div class="pop-up-message transition invisible hidden">
        <div class="message"></div>
      </div>
    `);
  }
  $(".pop-up-message .message").text(messageText);
  singleElementTransition(".pop-up-message", true);
  setTimeout(() => {
    singleElementTransition(".pop-up-message", false);
  }, 4000);
}

export function singleElementTransition(elementsSelector, shouldBeVisible) {
  if (shouldBeVisible && $(elementsSelector).hasClass("invisible")) {
    $(elementsSelector).removeClass("hidden");
    setTimeout(() => {
      $(elementsSelector).removeClass("invisible");
    }, 150);
  } else if (!shouldBeVisible && !$(elementsSelector).hasClass("invisible")) {
    $(elementsSelector).addClass("invisible");
    setTimeout(() => {
      $(elementsSelector).addClass("hidden");
    }, 300);
  }
}

export function isMobile() {
  try {
    document.createEvent("TouchEvent");
    return true;
  } catch (e) {
    return false;
  }
}

export function getBrowser(returnObject = false, returnTabs = true, checkOnWebsite = false) {
  var userAgent = navigator.userAgent;
  if (isWebVersion() && !checkOnWebsite) {
    return returnObject ? undefined : "";
  }

  if (userAgent.indexOf("Chrome") != -1) {
    if (returnObject && returnTabs) {
      return typeof chrome !== "undefined" && typeof chrome.tabs !== "undefined" ? chrome.tabs : undefined;
    } else if (returnObject) {
      return typeof chrome !== "undefined" ? chrome : undefined;
    } else {
      return "chrome";
    }
  } else if (userAgent.indexOf("Firefox") != -1 || userAgent.indexOf("Safari") != -1) {
    if (returnObject && returnTabs) {
      return typeof browser !== "undefined" && typeof browser.tabs !== "undefined" ? browser.tabs : undefined;
    } else if (returnObject) {
      return typeof browser !== "undefined" ? browser : undefined;
    } else {
      return userAgent.indexOf("Safari") != -1 ? "safari" : "firefox";
    }
  } else {
    return returnObject ? undefined : "";
  }
}

export function isWebVersion() {
  try {
    return window.location.href.includes("portable");
  } catch (e) {
    return false;
  }
}

export function timeout(ms, promise) {
  return new Promise(function(resolve, reject) {
    setTimeout(function() {
      reject(new Error("Timeout after " + ms + " ms"));
    }, ms);
    promise.then(resolve, reject);
  });
}

export function openTabOrUrl(tabId, url) {
  var tabsObject = getBrowser(true);
  if (!tabsObject) {
    if (!window.open(url)) {
      window.location.href = url;
    }
    return;
  }
  if (!tabId) {
    openUrl(url);
    return;
  }
  try {
    tabsObject
      .get(tabId)
      .then(data => {
        if (data) {
          if (data.url != url) {
            new CustomModal(
              "The content of the tab has changed. Please select if you wish to open the original URL or the current URL.",
              "",
              `<button id="open-old-tab-btn" class="gradient-btn">Open the original URL</button>
            <button id="open-new-tab-btn" class="gradient-btn">Open the current URL</button>`
            );
            $("#open-old-tab-btn").on("click", function() {
              openUrl(url);
              CustomModal.closeModal(this);
            });
            $("#open-new-tab-btn").on("click", function() {
              tabsObject.update(tabId, { active: true });
              CustomModal.closeModal(this);
            });
          } else {
            tabsObject.update(tabId, { active: true });
          }
        } else {
          openUrl(url);
        }
      })
      .catch(() => {
        openUrl(url);
      });
  } catch (error) {
    console.log(error);
    openUrl(url);
  }

  function openUrl(url) {
    try {
      tabsObject.query({}, function(tabData) {
        tabData = tabData.filter(tab => tab.url == url);
        if (tabData.length) {
          tabsObject.update(tabData[0].id, { active: true }).catch(_error => {
            if (!window.open(url)) {
              window.location.href = url;
            }
          });
        } else if (!window.open(url)) {
          tabsObject.create({ url, active: true });
        }
      });
    } catch (_error) {
      if (!window.open(url)) {
        window.location.href = url;
      }
    }
  }
}

export function bookmarkTabs(bookmarks, groupTitle) {
  var browserObject = getBrowser(true, false);
  if (browserObject && browserObject.permissions) {
    browserObject.permissions.request(
      {
        permissions: ["bookmarks"]
      },
      granted => {
        if (granted) {
          try {
            iterateBookmarks(browserObject, bookmarks, groupTitle);
            addPopupMessage(bookmarks.length > 1 ? "groupBookmark" : "tabBookmark", { group: groupTitle });
          } catch (error) {
            addPopupMessage(bookmarks.length > 1 ? "groupBookmarkError" : "tabBookmarkError");
          }
        } else {
          new CustomModal(VALIDATION_MESSAGES.bookmarksPermissionDenied);
        }
      }
    );
  } else {
    new CustomModal(VALIDATION_MESSAGES.bookmarkNotSupported);
  }

  function iterateBookmarks(browserObject, bookmarksData, groupTitle) {
    groupTitle = "TabCrunch | " + groupTitle;
    browserObject.bookmarks.search({ title: groupTitle }).then(bookmarks => {
      var bookmarkFolders = bookmarks.filter(bookmark => {
        return !bookmark.url;
      });
      if (!bookmarkFolders.length) {
        browserObject.bookmarks.create({ title: groupTitle }).then(newFolder => {
          for (var bookmarkData of bookmarksData) {
            createBookmarks(newFolder.id, bookmarkData);
          }
        });
      } else {
        for (var bookmarkData of bookmarksData) {
          createBookmarks(bookmarkFolders[0].id, bookmarkData);
        }
      }
    });
  }

  function createBookmarks(folderId, bookmarkData) {
    browserObject.bookmarks.search({ url: bookmarkData.url }, function(existingBookmark) {
      if (existingBookmark.length && existingBookmark.some(bookmark => bookmark.parentId == folderId)) {
        return;
      }
      browserObject.bookmarks.create({ url: bookmarkData.url, title: bookmarkData.title, parentId: folderId });
    });
  }
}

export function numberFormatter(number) {
  return number.toString().replace(/\B(?=(\d{3})+(?!\d))/g, ",");
}

export function addOpenAIMessage(type, containerSelector = "body") {
  if (type === "none") {
    return;
  }
  var messageText = POPUP_MESSAGES["openAIMinor"];
  var openAIStatusClass = "warning";
  if (type !== "" && type !== "minor") {
    messageText = POPUP_MESSAGES["openAIMajor"];
    openAIStatusClass = "error";
  }
  if (!$(".openai-message").length) {
    $(containerSelector).append(`
      <div class="openai-message transition invisible hidden">
        <div class="openai-${openAIStatusClass}">${messageText}</div>
      </div>
    `);
  }
  $("#bubble-info-holder").addClass("bubble-no-click");
  singleElementTransition(".openai-message", true);
}

export async function getOpenAIStatus() {
  try {
    var response = await fetch("https://status.openai.com/api/v2/status.json");
    var data = await response.json();
    return data.status.indicator;
  } catch (error) {
    return "";
  }
}

export function formatMinutes(minutes, isShortForm = true) {
  if (minutes <= 0) {
    return `0 ${isShortForm ? "min" : "minutes"}`;
  }

  const hours = Math.floor(minutes / 60);
  const remainingMinutes = minutes % 60;
  const minutesText = isShortForm ? "min" : remainingMinutes > 1 ? "minutes" : "minute";

  if (hours > 0) {
    return `${hours} ${isShortForm ? "hrs" : hours > 1 ? "hours" : "hour"}  ${remainingMinutes} ${minutesText}`;
  } else {
    return `${remainingMinutes} ${minutesText}`;
  }
}

export function escapeHTML(text) {
  return text
    ? text.replace(
        /[&<>'"]/g,
        tag =>
          ({
            "&": "&amp;",
            "<": "&lt;",
            ">": "&gt;",
            "'": "&#39;",
            '"': "&quot;"
          }[tag] || tag)
      )
    : "";
}

export function escapeSpecialCharacters(str) {
  var escapedStr = $("<div/>").attr("data-something", str)[0].outerHTML;
  escapedStr = escapedStr.substr(escapedStr.indexOf('"') + 1);
  escapedStr = escapedStr.substr(0, escapedStr.indexOf('"'));
  return escapedStr;
}

export function stripURL(url) {
  return url.split("#")[0].split("://")[1];
}

/**
 * Function that creates a custom version of the ordered list items
 * @param {Array} listItemContents contains the inner HTML of the list items
 * @param {Element} container (optional) the <ol> element
 * @param {Array} itemClasses (optional) the classes of the <li> element
 * @returns {void|string} if the container attribute is not given, returns the string represantation of the list items
 */
export function addCustomOrderedList(listItemContents, container = null, itemClasses = []) {
  var listItemsHTML = "";
  itemClasses = itemClasses.length ? 'class="' + itemClasses.join(" ") + '"' : "";

  for (var i in listItemContents) {
    var listItemCount = parseInt(i) + 1;
    var itemContent = listItemContents[i];

    listItemsHTML += `
      <li ${itemClasses}>
        <span class="custom-li-marker">${listItemCount}.</span>${itemContent}
      </li>
    `;
  }
  if (container) {
    container.append(listItemsHTML);
  } else {
    return listItemsHTML;
  }
}

export function updateFavourites(userId, data, checked, callback) {
  fetch(API_URL + "/favourite/" + userId + `?api_version=${API_VERSION}`, {
    method: "POST",
    headers: HEADERS,
    body: JSON.stringify(data)
  })
    .then(response => {
      if (response.ok) {
        var message = checked ? "tabFavouriteRemove" : "tabFavourite";
        callback();
        addPopupMessage(message);
      }
    })
    .catch(error => {
      console.log(error);
      addPopupMessage("tabFavouriteError");
    });
}

export function closeTabsFromButton(tabsData, userId, callback = undefined) {
  if (!tabsData.length) {
    addPopupMessage("tabCloseNotFound");
    return;
  }
  var tabIds = tabsData.map(tab => tab.id);
  fetch(API_URL + "/delete_tab/" + userId + `?api_version=${API_VERSION}`, {
    method: "POST",
    headers: HEADERS,
    body: JSON.stringify({ ids: tabIds.map(tab => tab.toString()) })
  })
    .then(response => {
      if (response.status >= 500) {
        throw new Error(VALIDATION_MESSAGES.serverError);
      }
      if (response.ok) {
        addPopupMessage("tabClose", { tabsCount: tabIds.length });
        closeTabsInBrowser();
      }
    })
    .catch(error => {
      new CustomModal(error.message, "", "");
    });

  function closeTabsInBrowser() {
    var tabsObject = getBrowser(true);
    if (tabsObject && tabsObject.remove) {
      try {
        tabsObject.query({}, function(currentTabs) {
          var currentUrlsMap = Object.values(currentTabs).reduce((acc, tab) => {
            if (!acc[tab.url]) {
              acc[tab.url] = [];
            }
            if (!tab.pinned) {
              acc[tab.url].push(tab.id);
            }
            return acc;
          }, {});
          var currentClosedTabIds = [];
          for (var tab of tabsData) {
            var urlIndexes = currentUrlsMap[tab.url];
            if (urlIndexes) {
              currentClosedTabIds.push(...urlIndexes);
            }
          }
          if (currentClosedTabIds.length) {
            tabsObject.remove(currentClosedTabIds);
          }
          if (callback) {
            callback();
          }
        });
      } catch (error) {
        console.log(error);
      }
    } else if (callback) {
      callback();
    }
  }
}

export function initActionsMenu() {
  $(".action-menu-btn").on("click", e => {
    e.stopPropagation();
    if ($("#actions-dropdown-container").hasClass("active")) {
      $("#actions-dropdown-container").removeClass("active");
      $(".action-menu-btn").attr("data-title", "Open action menu");
    } else {
      $("#actions-dropdown-container").addClass("active");
      $(".action-menu-btn").attr("data-title", "Close action menu");
      $("body").on("click", toggleActionMenu);

      function toggleActionMenu(e) {
        e.preventDefault();
        e.stopPropagation();
        if (
          !$(e.target).hasClass("action-menu-btn") &&
          !$(e.target)
            .parent()
            .hasClass("action-menu-btn")
        ) {
          $("#actions-dropdown-container").removeClass("active");
          $("body").off("click", toggleActionMenu);
        }
      }
    }
  });
  singleElementTransition(".action-menu-btn", true);
}

export function initButtonAdditionalContainerClose(buttonClass, containerSelector) {
  $("body").on("click", function(e) {
    var additionalContainerParents = Object.values($(e.target).parents()).filter(element => {
      return $(element).hasClass("btn-additional-container");
    });
    if (!additionalContainerParents.length && !$(e.target).hasClass(buttonClass)) {
      singleElementTransition(containerSelector, false);
      $("body").off("click");
      $("#move-container-holder").addClass("hidden");
    }
  });
}

export function toggleShareLinkContainer(shareUrl) {
  singleElementTransition("#edit-container", false);
  singleElementTransition("#share-container", true);
  initButtonAdditionalContainerClose("share-btn", "#share-container");
  $("#share-link").val(shareUrl);
  $("#copy-share-btn").on("click", function() {
    navigator.clipboard
      .writeText(shareUrl)
      .then(function() {
        addPopupMessage("shareLink");
        $("body").off("click");
        singleElementTransition("#share-container", false);
      })
      .catch(function(_error) {
        addPopupMessage("shareLinkError");
      });
  });
  $("#close-share-btn").on("click", function() {
    singleElementTransition("#share-container", false);
  });
}

export function preloader(enabled, container, text = "") {
  if (enabled) {
    $(container).append(`
      <div class="preloader-container">
        <div class="preloader">
          <div class="dot"></div>
          <div class="dot"></div>
          <div class="dot"></div>
          <div class="dot"></div>
          <div class="dot"></div>
        </div>
        <span class="preloader-text" class="${text ? "" : "hidden"}">${text}</span>
      </div>
    `);
  } else {
    $(container)
      .find(".preloader-container")
      .remove();
  }
}

export function startPreloader(containerToHide, preloaderContainer, text = "") {
  singleElementTransition(containerToHide, false);
  setTimeout(() => {
    preloader(true, preloaderContainer, text);
  }, 300);
}
