import { API_VERSION, URL_REG_EXPRESSION, API_URL, HEADERS, VALIDATION_MESSAGES } from "./config.js";
import { getGroupsData, transition, getUserData, startAnalysis } from "./panel.js";
import {
  getBrowser,
  addPopupMessage,
  singleElementTransition,
  isWebVersion,
  openTabOrUrl,
  startPreloader,
  preloader
} from "./common.js";
import { InstructionsModal } from "./instructions-modal.js";
import { TourMessageManager } from "./tour.js";

var uniqueId = undefined;
var groups = [];
var userData = {};
var analysedUrls = [];

var callback = function() {
  fetch(API_URL + "/status/" + uniqueId + `?api_version=${API_VERSION}`, { method: "GET" })
    .then(response => response.json())
    .then(data => {
      analysedUrls = data && data.urls ? data.urls.map(url => url.url) : [];
      groups = typeof data == "object" ? getGroupsData(data) : [];
      if (typeof groups === "string") {
        groups = [];
      }
      initPage();
    })
    .catch(err => {
      console.log(err);
      new CustomModal(VALIDATION_MESSAGES.serverError, "", "");
    });
};

export function initImportPage() {
  userData = getUserData();
  uniqueId = userData["userId"];
  callback();
}

function initPage() {
  for (var group of groups) {
    for (var tab of group.tabs) {
      var dateFormated = "";
      var dateFragments = tab.created ? tab.created.split("-") : "";
      if (dateFragments.length > 1) {
        dateFragments[2] = dateFragments[2].split("T")[0];
        dateFormated = dateFragments[2].split("T")[0] + "." + dateFragments[1] + "." + dateFragments[0];
      }
      tab.created = dateFormated;
      tab.category = group.name;
    }
  }
  initActionOptions();
  initExportCategories();
  initExportButton();

  var urlParams = new URLSearchParams(window.location.search);
  if (urlParams.get("p") == "import") {
    $("#action-import").click();
  } else {
    $("#main-import").removeClass("transparent");
  }

  document.addEventListener("paste", event => handleTabsImport(event));
  $("#import-tabs-area").on("keyup", event => handleTabsImport(event, true));

  handleFileUpload();
  initImportButtons();
  initCancelButton();
}

function initActionOptions() {
  $("#action-import").on("click", function() {
    setTimeout(() => {
      $("#section-title").text("Import tabs");
      $("#main-import").removeClass("transparent");
    }, 350);
    transition(
      "#action-type-container, #text-header",
      "#action-type-container",
      "#import-options, #text-header, #header-back-btn"
    );
    TourMessageManager.initByPage();
  });

  $("#action-export").on("click", function() {
    setTimeout(() => {
      $("#section-title").text("Export tabs from:");
    }, 350);
    transition(
      "#action-type-container, #text-header",
      "#action-type-container, #header-back-btn",
      "#export-container, #text-header"
    );
    if (typeof groups === "object" && Object.keys(groups).length > 1) {
      singleElementTransition("#export-select-all", true);
    }
  });

  $("#text-header .back-btn").on("click", function() {
    transition("#import-options, #text-header", "#import-options", "#action-type-container");
  });

  $("#instruction-link").on("click", () => new InstructionsModal("import"));
}

function initExportCategories() {
  var categoriesHTML = "";
  for (var group of groups) {
    categoriesHTML += `
      <div class="checkbox-btn" data-category="${group.id}">
        <div class="checkbox-outline"></div>
        <span class="checkbox-title">${group.name}</span>
      </div>
    `;
  }
  if (groups.length) {
    $("#categories").append(categoriesHTML);
    $("#categories .checkbox-btn").each(function() {
      var button = $(this);
      $(this).on("click", function(e) {
        e.stopPropagation();
        button.toggleClass("checked");
        $(this)
          .find("input")
          .prop("checked", button.hasClass("checked"));
        var shouldBeChecked = $("#categories .checkbox-btn.checked").length === groups.length;
        $("#export-select-all")
          .attr("data-mode", shouldBeChecked ? "deselect" : "select")
          .text(shouldBeChecked ? "Deselect all" : "Select all");
      });
    });
  }

  $("#export-select-all").on("click", function() {
    var shouldBeChecked = $(this).attr("data-mode") == "select";
    $(this).addClass("transparent");
    $(this).attr("data-mode", shouldBeChecked ? "deselect" : "select");
    setTimeout(() => {
      $(this).text(shouldBeChecked ? "Deselect all" : "Select all");
      $(this).removeClass("transparent");
    }, 300);
    $("#categories .checkbox-btn").each(function() {
      if (shouldBeChecked) {
        $(this).addClass("checked");
      } else {
        $(this).removeClass("checked");
      }
    });
  });
}

function initExportButton() {
  $("#export-format .format").on("click", function() {
    var selectedCategories = [];
    $(".checkbox-btn.checked").each(function() {
      selectedCategories.push($(this).attr("data-category"));
    });
    if (!selectedCategories.length) {
      addPopupMessage("noCategoriesSelected");
      return;
    }
    var selectedFormat = $(this).attr("data-format");
    var groupNames = "";
    var groupsData = groups.filter(group => {
      if (selectedCategories.indexOf(group.id.toString()) > -1) {
        groupNames += group.name + ",";
        return true;
      }
      return false;
    });
    groupNames = groupNames.slice(0, -1);
    if (groupNames.length > 100) {
      groupNames = groupNames.slice(0, 100) + " and other";
    }

    downloadExportedFile(selectedFormat, groupsData, groupNames);
  });

  $("#export-btn").on("click", function() {
    $(this).toggleClass("active");
  });
}

export function downloadExportedFile(selectedFormat, groupsData, fileName) {
  var columns =
    groupsData.length > 1
      ? ["title", "url", "category", "readTime", "created"]
      : ["title", "url", "readTime", "created"];
  var columnNames =
    groupsData.length > 1
      ? ["Title", "URL", "Group", "Reading time", "Date added"]
      : ["Title", "URL", "Reading time", "Date added"];
  var content =
    selectedFormat == "csv"
      ? getExportCSV(groupsData, columnNames, columns)
      : getExportJSON(groupsData, columnNames, columns);
  var contentObject = new Blob(["\ufeff", content], { type: `text/${selectedFormat}` });
  var url = URL.createObjectURL(contentObject);
  var link = document.createElement("a");
  link.setAttribute("href", url);
  link.setAttribute("target", "__blank");
  link.setAttribute("download", `TabCrunch - ${fileName}.${selectedFormat}`);
  document.body.appendChild(link);
  if (!isWebVersion()) {
    // Fix for issue with downloading blobs in Safari extensions
    var iframe = document.querySelector("#download-link-container");
    var iframeDocument = iframe.contentDocument || iframe.contentWindow.document;
    iframeDocument.body.appendChild(link);
  }
  setTimeout(() => {
    link.click();
    link.remove();
  }, 500);
}

function getExportCSV(groupsData, columnsRenamed, columns) {
  return (
    columnsRenamed.join(",") +
    "\n" +
    groupsData
      .map(group => {
        var result = [];
        for (var tab of group.tabs) {
          var tabsForCategory = [];
          for (var column of columns) {
            tabsForCategory.push(tab[column] ? '"' + tab[column].toString().replace(/"/g, '""') + '"' : '""');
          }
          result.push(tabsForCategory);
        }
        return result.join("\n");
      })
      .join("\n")
  );
}

function getExportJSON(groupsData, columnsRenamed, columns) {
  var json = {};
  for (var group of groupsData) {
    json[group.name] = [];
    for (var tab of group.tabs) {
      var tabData = {};
      for (var i in columns) {
        tabData[columnsRenamed[i]] = tab[columns[i]];
      }
      json[group.name].push(tabData);
    }
  }
  return JSON.stringify(json);
}

function handleFileUpload() {
  $("#bulk-file-input").on("change", function() {
    var file = $("#bulk-file-input").prop("files")[0];
    const allowedTypes = ["text/csv", "text/html", "text/plain"];
    $("#bulk-file-input").prop("files")[0];
    CustomModal.closeModal();

    if (!allowedTypes.includes(file.type)) {
      $("#bulk-file-input").val("");
      new CustomModal(VALIDATION_MESSAGES.fileExtension, "", "");
      return;
    }

    var reader = new FileReader();
    reader.readAsText(file);
    reader.onload = function(event) {
      var text = event.target.result;
      var lines = text.split("\n");
      if (lines) {
        setTimeout(() => {
          $("#section-title").text("Imported tabs:");
          $("#text-header").addClass("centered");
        }, 350);
        transition(
          "#import-options, #text-header",
          "#import-options, #header-back-btn",
          "#tabs-container, #text-header"
        );
        var urls = text.match(URL_REG_EXPRESSION);
        addTabs(Object.values(urls));
      }
    };
  });
}

function addTabs(urls) {
  var newUrls = [];
  var alreadyAnalysedUrls = [];
  $("#imported-tabs").empty();
  urls.forEach(url => {
    if (analysedUrls.indexOf(url) == -1 && newUrls.indexOf(url) == -1) {
      newUrls.push(url);
    } else if (analysedUrls.indexOf(url) > -1 && alreadyAnalysedUrls.indexOf(url) == -1) {
      alreadyAnalysedUrls.push(url);
    }
  });
  addTabByType(newUrls, "new");
  if (!$("#imported-tabs #no-new-message").length) {
    addTabByType(alreadyAnalysedUrls, "analysed");
  }
  singleElementTransition("#import-buttons .btn-analyse", newUrls.length == 0 ? false : true);

  function addTabByType(urls, type) {
    if (urls.length == 0 && type !== "new") {
      return;
    }
    var container = $("#imported-tabs");
    var sectionTitle = "";
    if (type === "new") {
      sectionTitle = "Tabs ready for analysis";
    } else if (type === "analysed") {
      sectionTitle = "Previously analysed tabs";
    }
    if (urls.length == 0 && type === "new") {
      if (!$("#imported-tabs #no-new-message").length) {
        container.append(
          `<p id="no-new-message" class="list-group">All these tabs have already been analysed. No new tabs to analyse.</p>`
        );
      }
      return;
    } else {
      container.append(`<div class="list-group ${type}-group">${sectionTitle}:</div>`);
    }
    for (const url of urls) {
      var formattedUrl = url.trim().replace(/https?:\/\//, "");
      if (formattedUrl.charAt(formattedUrl.length - 1) == "/") {
        formattedUrl = formattedUrl.slice(0, -1);
      }
      container.append(`
      <div class="${type}">
        <button class="close-btn icon-btn"></button>
        <a href="${url}" class="link" target="_blank">${formattedUrl}</a>
      </div>`);
    }
  }
}

function initImportButtons() {
  $("#imported-tabs").on("click", "a", function(e) {
    e.stopPropagation();
    var tabsObject = getBrowser(true);
    if (tabsObject) {
      var url = $(this).attr("href");
      e.preventDefault();
      openTabOrUrl(0, url);
    }
  });

  $("#imported-tabs").on("click", ".close-btn", function() {
    var tabRow = $(this).parent()[0];
    var tabType = $(tabRow).hasClass("new") ? "new" : "analysed";
    $("#tabs-container .invalid").addClass("invisible");
    if ($(`#imported-tabs .${tabType}`).length <= 1) {
      $(`#imported-tabs .${tabType}-group`).remove();
    }
    $(tabRow).remove();
    if (!$("#imported-tabs .new, #imported-tabs .analysed").length) {
      $(".btn-analyse").addClass("invisible");
    }
  });

  $("#import-buttons").on("click", ".btn-analyse", function() {
    var tabList = [];
    var validUrls = [];
    var textareaData = $("#imported-tabs .new");
    if (textareaData) {
      textareaData.each(function() {
        var tab = $(this)
          .find("a")
          .attr("href")
          .trim();
        if (tab && tab.match(URL_REG_EXPRESSION)) {
          validUrls.push(tab);
        }
      });
    } else {
      $("#imported-tabs p").each(function() {
        var currentUrl = $(this).text();
        validUrls.push(currentUrl);
      });
    }
    if (validUrls) {
      for (var url of validUrls) {
        tabList.push({
          tabTitle: "",
          tabId: CryptoJS.MD5(url).toString(),
          tabUrl: url,
          tabContent: "",
          tabSource: "manual"
        });
      }
      userData = getUserData();
      uniqueId = userData["userId"];
      startPreloader("main", "body");
      startAnalysis(() => {
        fetch(API_URL + "/tabs/" + uniqueId + `?api_version=${API_VERSION}`, {
          method: "POST",
          headers: HEADERS,
          body: JSON.stringify(tabList)
        })
          .then(response => {
            setTimeout(() => {
              preloader(false, "body");
              if (response.status !== 200) {
                new CustomModal(VALIDATION_MESSAGES.serverError, "", "");
                return;
              }
              addPopupMessage(VALIDATION_MESSAGES.importSuccess);
              TourMessageManager.initByStep(["import", "importMobile"], "finish");
              navigateToPage("analyse.html", true);
            }, 2000);
          })
          .catch(() => {
            new CustomModal(VALIDATION_MESSAGES.serverError, "", "");
          });
      });
    }
  });
}

function initCancelButton() {
  $(".button-wrapper").on("click", ".cancel-btn", function() {
    $("#bulk-file-input").val("");
    $("#import-tabs-area").val("");
    if ($(this).attr("id") == "cancel-import") {
      $("#import-tabs-area").on("keyup", event => handleTabsImport(event, true));
      setTimeout(() => {
        $("#section-title").text("Import tabs");
        $("#text-header").removeClass("centered");
        $("#imported-tabs").empty();
      }, 300);
      transition("#tabs-container, #text-header", "#tabs-container", "#import-options, #text-header, #header-back-btn");
    } else {
      transition(
        "#export-container, #text-header, #export-select-all",
        "#export-container, #export-select-all",
        "#action-type-container"
      );
      $("#categories .checkbox-btn").removeClass("checked");
      setTimeout(() => {
        $("#export-select-all").attr("data-mode", "select");
        $("#export-select-all").text("Select all");
      }, 300);
    }
    setTimeout(() => {
      TourMessageManager.initByPage();
    }, 800);
  });
}

function handleTabsImport(event, isFromGoogleClipboard = false) {
  if (isFromGoogleClipboard) {
    var isStringData = true;
    var parsed = $("#import-tabs-area")
      .val()
      .split(" ");
  } else {
    $("#import-tabs-area").off("keyup");
    var clipboardData = event.clipboardData ? event.clipboardData : window.clipboardData;
    var pasteHtml = clipboardData.getData("text/html");
    var isStringData = !pasteHtml;
    var parsed = !isStringData ? $.parseHTML(pasteHtml) : clipboardData.getData("text").split("\n");
  }
  var result = [];
  for (var element of parsed) {
    if (isStringData && element.match(/\d\.( )?/)) {
      element = element.split(".");
      element.shift();
      element = element.join(".").trim();
      if (element) {
        result.push(element);
      }
    } else if (isStringData && /^https?:\/\/+/.test(element.trim())) {
      result.push(element);
    } else if (!isStringData) {
      var that = $(element);
      var anchorElements = that.find("a");
      if ((!anchorElement || !anchorElement.length) && that.is("a")) {
        anchorElements = that;
      }
      if (anchorElements.length) {
        for (var anchorElement of anchorElements) {
          var link = $(anchorElement).attr("href");
          if (link) {
            result.push(`${link.trim()}`);
          }
        }
      }
    }
  }
  addTabs(result);
  if (result.length) {
    setTimeout(() => {
      $("#section-title").text("Imported tabs:");
      $("#text-header").addClass("centered");
    }, 350);
    transition("#import-options, #text-header", "#import-options, #header-back-btn", "#tabs-container, #text-header");
  } else {
    addPopupMessage("importError");
  }
  event.preventDefault();
}
