import { updateFavourites } from "./common.js";
import { getUserData } from "./panel.js";

window.LiveTable = class LiveTable {
  /**
   * Constructor
   * All configurations can be set from here
   */
  constructor(selector, data, parameters = {}) {
    // Default parameters
    this.filter = [];
    this.sorting = "";
    this.sortingSecondary = null;
    this.searchType = "default";
    this.order = "default";
    this.columns = [];
    this.noSorting = [];
    this.instance = "liveTable";
    this.noData = "No results found.";
    this.minTableCell = [];
    this.filterIgnore = [];
    this.modifyRowAfterRender = () => {};
    this.callOnSort = () => {};
    this.callOnFilter = () => {};

    this.data = data.filter(e => e);
    this.selector = selector;
    for (var key in parameters) {
      this[key] = parameters[key];
    }
    if (this.sorting === "") {
      this.sorting = this.columns[0].name === "#" ? this.columns[1].name : this.columns[0].name;
    }
    this.filterIgnore.push("#", "x", "color", "favourite");
    this.prepareSelector();
    this.render();
  }

  /**
   * Function that adds classes and liveTable instance to this.selector when the constructor is called
   */
  prepareSelector() {
    $(this.selector)
      .addClass("live-table-wrapper")
      .data("liveTableInstance", this);
  }

  /**
   * Function that sets new data and re-renders the table
   */
  setData(data) {
    if (this.data.length === data.length) {
      return;
    }
    this.data = data;
    this.render();
  }

  /**
   * Function that sets the query filter rule and re-renders the table
   * @param {string|array} filter search filter
   */
  setFilter(filter) {
    this.filter = filter;
    this.render();
  }

  /**
   * Function that sets the sorting rule and re-renders the table
   */
  setSorting(sorting) {
    if (this.sorting === sorting) {
      this.order = this.order === "asc" ? "desc" : "asc";
    } else {
      this.order = "asc";
      for (var column of this.columns) {
        if (column.name === sorting) {
          this.order = column.default || "asc";
          break;
        }
      }
    }
    this.sorting = sorting;
    this.render();
  }

  /**
   * Function used to search for certain row and remove it
   */
  removeRow(searchColumn, searchValue) {
    for (var row in this.data) {
      if (this.data[row][searchColumn]["value"] == searchValue) {
        this.data.splice(row, 1);
        this.render();
        return;
      }
    }
  }

  /**
   * Function used to update data in specific row
   */
  updateRowData(column, searchColumn, searchValue, newText = null, newValue = null, sortValue = null, csvValue = null) {
    for (var row in this.data) {
      if (this.data[row][searchColumn]["value"] == searchValue) {
        if (newValue !== null) {
          this.data[row][column]["value"] = newValue;
        }
        if (newText !== null) {
          this.data[row][column]["text"] = newText;
        }
        if (sortValue !== null) {
          this.data[row][column]["sortValue"] = sortValue;
        }
        if (csvValue !== null) {
          this.data[row][column]["csvValue"] = csvValue;
        }
        break;
      }
    }
  }

  /**
   * Function that renders the table head
   */
  outputHead() {
    var that = this;
    return (
      `<thead><tr>` +
      this.columns
        .map(column => {
          if (column.name === "#") {
            return '<th class="min-table-cell">#</th>';
          }
          if (column.name === "color-no-event") {
            return '<th class="min-table-cell"></th>';
          }
          var thContent = column.text;
          var className = "";
          if (this.noSorting.indexOf(column.name) === -1) {
            thContent = `<a href="">${column.text}</a>`;
            className = column.name === that.sorting ? `sort ${that.order} active` : `sort desc`;
          }
          if (this.minTableCell.indexOf(column.name) > -1) {
            className += " min-table-cell";
          }
          return `<th id="th-${column.name}" class="${column.name} ${className}">${thContent}</th>`;
        })
        .join("") +
      `</tr></thead>`
    );
  }

  /**
   * Function that renders the table body
   */
  outputBody() {
    var that = this;
    // Filtering by searched word + sorting
    var dataFiltered = this.data;
    dataFiltered.sort(this.sortingComparator(that));
    dataFiltered.forEach(this.searchProcess(that));
    if (this.searchType == "content") {
      localStorage.setItem("tabSearch", !this.filter || !this.filter.length || !this.filter[0] ? "" : this.filter[0]);
      this.callOnFilter(dataFiltered);
    }

    // Output
    var counter = 1;
    var toReturn = "";
    if (!dataFiltered.some(row => row.__visible)) {
      toReturn += `<tr class="noData"><td colspan="${this.columns.length}">` + this.noData + `</td></tr>`;
    }
    for (var row of dataFiltered) {
      var linkAttribute = row.__link ? `data-href="${row.__link}"` : "";
      var __class = row.__class || "";
      var rowClasses = row.__link ? `page-link ${__class}` : __class;
      toReturn +=
        "<tr " +
        (row.__visible ? "" : `style="display:none"`) +
        (row.__link ? ` ${linkAttribute}` : "") +
        (rowClasses.length ? ` class="${rowClasses}"` : "") +
        ">";
      var firstCellLink = false;
      for (var column of this.columns) {
        if (column.name === "#") {
          toReturn += `<td class="min-table-cell">${counter}</td>`;
        } else if (column.name === "x") {
          var closeTabButtonText = column.title ? `data-title="${column.title}"` : "";
          var btnType = this.instance && this.instance == "favourite" ? "remove-favourite" : "close";
          toReturn += `<td ${closeTabButtonText} class="min-table-cell"><button class="${btnType}-btn icon-btn"></button></td>`;
        } else if (column.name === "color") {
          var tabId = row["tabId"] ? `data-tab="${row["tabId"].value}"` : "";
          var websiteGroup = row["website-group"] ? `data-website="${row["website-group"].value}"` : "";
          toReturn += `
              <td class="min-table-cell color">
                <div class="topic-color-container">
                  <span class="group-color" ${tabId} ${websiteGroup} style="background-color: ${
            row[column.name].value
          }" data-title="Move tab to a different group"></span>
                </div>
              </td>`;
        } else if (column.name == "favourite") {
          var tabId = row["tabId"] ? `data-tab="${row["tabId"].value}"` : "";
          var isFavourite = row[column.name] && row[column.name].value;
          var imageSrc = isFavourite ? "images/favourite_remove.svg" : "images/favourite_dark.svg";
          var hoverTitile = isFavourite ? "Remove from favourite tabs" : "Add to favourite tabs";
          toReturn += `
            <td class="min-table-cell favourite">
              <div class="favourite-wrapper">
                <img class="favourite-btn icon-btn" ${tabId} src="${imageSrc}" data-title="${hoverTitile}">
              </div>
            </td>`;
        } else if (column.name == "searchSummary") {
          var href = row[column.name].text;
          var cellText = row[column.name].value ? "View summary" : "Request summary";
          toReturn += `
            <td class="min-table-cell">
              <a class="link page-link" href="${href}">${cellText}</a>
            </td>`;
        } else {
          var contents = row[column.name].text
            ? row[column.name].text.toString()
            : row[column.name].text === 0
            ? "0"
            : "";
          if (row.__link && contents.indexOf("<") === -1 && !firstCellLink) {
            contents = `<span class="link-style">${contents}</span>`;
            firstCellLink = true;
          } else if (row.__link && !firstCellLink) {
            firstCellLink = true;
          }
          if (row[column.name]["colspan"] === 0) {
            continue;
          }
          toReturn +=
            '<td class="' +
            (row[column.name]["class"] ? row[column.name]["class"] : "shortStrings") +
            " " +
            (column.name === this.sorting ? "active" : "") +
            '"' +
            (row[column.name]["colspan"] ? ' colspan="' + row[column.name]["colspan"] + '"' : "") +
            ">" +
            contents +
            "</td>";
        }
      }
      toReturn += "</tr>";
      counter++;
    }
    return toReturn;
  }

  initEvents() {
    var that = this;
    // Filtering by searched word + sorting
    var dataFiltered = this.data;
    dataFiltered.sort(this.sortingComparator(that));
    dataFiltered.forEach(this.searchProcess(that));

    $(this.selector)
      .find("tbody tr")
      .each(function(i, rowElement) {
        var row = dataFiltered[i];
        if (!row) {
          return;
        }
        $(rowElement)
          .find("td")
          .each(function(j, cellElement) {
            var column = that.columns[j].name;
            if (!column) {
              return;
            }
            if (row[column] && $(cellElement).hasClass("favourite")) {
              $(cellElement).on("click", function() {
                that.checkElement("favourite", row[column].value, row["tabId"].value);
              });
            }
            if (row[column] && row[column].__onClick) {
              if (!$(cellElement).hasClass("color")) {
                $(cellElement).on("click", function(e) {
                  row[column].__onClick(this, e);
                });
              } else {
                $(cellElement)
                  .find(".group-color")
                  .on("click", function(e) {
                    row[column].__onClick(this, e);
                  });
              }
            }
          });
      });
  }

  /**
   * Function that renders the whole live table
   */
  render() {
    $(this.selector).html(
      `<div id="` +
        this.instance +
        `"><table class="table main-data theadFix">` +
        this.outputHead() +
        "<tbody>" +
        this.outputBody() +
        `</tbody></table></div>`
    );
    if ($(this.selector).siblings(".live-table-footer").length) {
      $(this.selector)
        .siblings(".live-table-footer")
        .find(".pagination")
        .remove();
    }
    var that = this;
    $(this.selector)
      .find("th a")
      .each(function() {
        $(this).on("click", function(event) {
          var columnName = $(this)
            .parent()
            .attr("id")
            .replace("th-", "");
          event.stopPropagation();
          that.setSorting(columnName);
          that.callOnSort(that.data);
          return false;
        });
      });
    this.initEvents();

    this.modifyRowAfterRender(this);
  }

  /**
   * Closure returning the sorting comparator function
   * @private
   */
  sortingComparator(that) {
    var compareParameters = function(sortingParameter, a, b) {
      // Sorting for numbers
      var aValueKey = typeof a[sortingParameter].sortValue === "undefined" ? "value" : "sortValue";
      var bValueKey = typeof b[sortingParameter].sortValue === "undefined" ? "value" : "sortValue";
      if (!isNaN(a[sortingParameter][aValueKey]) && !isNaN(b[sortingParameter][bValueKey])) {
        if (that.order === "asc") {
          return a[sortingParameter][aValueKey] - b[sortingParameter][bValueKey];
        }
        return b[sortingParameter][bValueKey] - a[sortingParameter][aValueKey];
      }
      // Sorting for strings
      var aValue = a[sortingParameter][aValueKey].toString().toLowerCase();
      var bValue = b[sortingParameter][bValueKey].toString().toLowerCase();
      if (aValue === bValue) {
        return 0;
      }
      if (that.order === "asc") {
        return aValue > bValue ? 1 : -1;
      }
      return aValue > bValue ? -1 : 1;
    };
    return function(a, b) {
      var result = compareParameters(that.sorting, a, b);
      if (result === 0 && that.sortingSecondary) {
        return compareParameters(that.sortingSecondary, a, b);
      }
      return result;
    };
  }

  /**
   * Closure returning the search processing function
   * @private
   */
  searchProcess(that) {
    if (that.searchType == "default") {
      return getDefaultSearcher;
    }
    if (that.searchType == "content") {
      return getContentSearcher;
    }

    function getDefaultSearcher(row) {
      var visible = 0;
      var isVisible = false;
      var regex = new RegExp("[^A-Za-zа-яА-ЯÀ-ÖØ-öø-ÿ]");
      for (var filterWord of that.filter) {
        if (filterWord === "") {
          visible++;
          isVisible = true;
        } else {
          for (var key in row) {
            var searchString = row[key].searchValue || row[key].value;
            if (
              that.filterIgnore.indexOf(key) === -1 &&
              searchString &&
              searchString
                .toString()
                .toLowerCase()
                .indexOf(filterWord.toLowerCase()) > -1
            ) {
              var rowValue = searchString.toString().toLowerCase();
              if (filterWord == null) {
                filterWord = "";
              }
              var index = rowValue.indexOf(filterWord.toLowerCase());
              if (index == 0 || regex.test(filterWord[0])) {
                visible++;
                isVisible = true;
              } else {
                var result = regex.test(rowValue[index - 1]);
                visible += Number(result);
                isVisible = result;
              }
              if (isVisible) {
                break;
              }
            }
          }
        }
      }
      row.__visible = visible === that.filter.length;
    }

    function getContentSearcher(row) {
      if (!that.filter || !that.filter.length || !that.filter[0]) {
        row.__visible = false;
        return;
      }
      var filter = that.filter[0];
      var searchWords = filter.split(/\s+/);
      var isMatch = searchWords.every(
        word =>
          (row.title && row.title.searchValue && row.title.searchValue.toLowerCase().includes(word)) ||
          (row.domain && row.domain.searchValue && row.domain.searchValue.toLowerCase().includes(word)) ||
          (row.content && row.content.searchValue && row.content.searchValue.toLowerCase().includes(word))
      );
      row.__visible = isMatch;
    }
  }

  checkElement(name, checked, id) {
    if (name === "favourite") {
      var userId = getUserData("userId");
      var newValue = checked ? 0 : 1;

      var callback = () => {
        this.updateRowData("favourite", "tabId", id, null, newValue);
        this.render();
      };
      updateFavourites(userId, { tabId: id }, checked, callback);
    }
  }
};
