import { isMobile, numberFormatter, formatMinutes } from "./common.js";
import { API_VERSION, API_URL, VALIDATION_MESSAGES } from "./config.js";
import { getGroupsData, inlineLiveSearchFormEvent, getUserData } from "./panel.js";

var uniqueId = undefined;
var liveTable = undefined;
var columns = undefined;
var columnsRenamed = {
  title: "Groups",
  readTime: "Reading time",
  tabsCount: "Tabs count",
  summary: "Summary",
  languages: "Languages"
};
var groups = [];
var userData = {};

var callback = function() {
  fetch(API_URL + "/status/" + uniqueId + `?api_version=${API_VERSION}`, { method: "GET" })
    .then(response => response.json())
    .then(data => getGroupsData(data))
    .then(data => {
      groups = typeof data == "object" ? data : [];
      initPage();
    })
    .catch(() => {
      new CustomModal(VALIDATION_MESSAGES.serverError, "", "");
    });
};

export function initStatisticsPage() {
  userData = getUserData();
  uniqueId = userData["userId"];
  callback();
}

function initPage() {
  initTable();
  initCharts();
  $("#table-holder").removeClass("transparent");
  $(".transition-container").removeClass("transparent");
  $("#search-tab-field").on("keyup", function() {
    inlineLiveSearchFormEvent(this, false);
  });
  $("main").tooltip();
  $("#statistics-main").removeClass("transparent");
}

function initTable() {
  var tableData = [];
  if (!groups || !groups.length) {
    $("#table-controls").addClass("hidden");
    $("#search-container").remove();
    return;
  }
  for (var i in groups) {
    var group = groups[i];
    var webistes = [];
    var languages = [];
    for (var tab of group.tabs) {
      var domain = tab.url.indexOf("://") > -1 ? tab.url.split("/")[2] : tab.url.split("/")[0];
      domain = domain.replace(/https?:\/\//, "");
      if (webistes.indexOf(domain) == -1) {
        webistes.push(domain);
      }
      if (tab.language && languages.indexOf(tab.language) == -1) {
        languages.push(tab.language);
      }
    }
    groups[i].websitesCount = webistes.length;
    groups[i].languages = languages
      .sort()
      .join(", ")
      .toUpperCase();
    var rowData = {
      groupId: initRowCell(group.id, "groupId"),
      "color-no-event": initRowCell(group.color, "color-no-event"),
      "#": i,
      title: initRowCell(group.name, "title"),
      tabsCount: initRowCell(group.tabs.length, "tabsCount"),
      readTime: initRowCell(group.readTime, "readTime"),
      summary: initRowCell(group.summary, "summary"),
      languages: initRowCell(group.languages, "languages"),
      __link: `topic.html?id=${group.id}&redirect=1`
    };
    tableData.push(rowData);
  }
  columns = !isMobile()
    ? ["color-no-event", "#", "title", "summary", "readTime", "tabsCount", "languages"]
    : ["color-no-event", "#", "title", "readTime", "tabsCount", "languages"];
  columns = columns.map(column => {
    return {
      name: column,
      text: columnsRenamed[column] ? columnsRenamed[column] : column,
      default: "desc"
    };
  });
  liveTable = new LiveTable(".live-table", tableData, {
    columns: columns,
    sorting: "readTime",
    noSorting: ["color-no-event", "summary"]
  });
  $("#search-tab-field").attr("placeholder", "Search groups");
  var search = $("#search-tab-field").val();
  $("#clear-search-btn").toggleClass("invisible", !search);
  if (search) {
    liveTable.setFilter([search], true);
  }
  $("#clear-search-btn").on("click", function() {
    var inputElement = $(this)
      .parents(".search-container")
      .find("input");
    inputElement.val("");
    liveTable.setFilter([inputElement.val()], true);
    $("#clear-search-btn").addClass("invisible");
  });
}

function initRowCell(value, column) {
  if (column == "color-no-event") {
    return {
      value: value,
      text: `<div class="topic-color-container">
                <span class="group-color" style="background-color: ${value}" data-title="Group colour"></span>
            </div>`
    };
  } else if (column == "summary") {
    return {
      value: value,
      text: value
    };
  } else if (column == "readTime") {
    return {
      value: value,
      text: value ? formatMinutes(value) : "N/A",
      class: column
    };
  } else if (column == "tabsCount") {
    return {
      value: value,
      text: numberFormatter(value),
      class: column
    };
  } else {
    if (!value) {
      value = "N/A";
    }
    return {
      value: value,
      text: value,
      class: column != "title" ? column : "group-title"
    };
  }
}

function initCharts() {
  var tabChartData = [];
  var readTimeChartData = [];
  var websitesChartData = [];
  for (var group of groups) {
    readTimeChartData.push({ id: group.name, value: group.readTime });
    tabChartData.push({ id: group.name, value: group.tabs.length });
    websitesChartData.push({ id: group.name, value: group.websitesCount });
  }
  Charts.initPieChart(readTimeChartData, "time-chart", "Reading time", "reading-time");
  Charts.initPieChart(tabChartData, "tabs-chart", "Tabs per group", "tabs-count");
  Charts.initPieChart(websitesChartData, "websites-chart", "Websites per group", "websites-count");
}
