import { API_VERSION, API_URL, HEADERS, VALIDATION_MESSAGES } from "./config.js";
import { getGroupsData, getUserData } from "./panel.js";
import {
  isMobile,
  getBrowser,
  isWebVersion,
  bookmarkTabs,
  addPopupMessage,
  formatMinutes,
  escapeHTML,
  openTabOrUrl,
  addCustomOrderedList,
  closeTabsFromButton,
  initActionsMenu,
  toggleShareLinkContainer
} from "./common.js";
import { TourMessageManager } from "./tour.js";

var uniqueId = undefined;
var urlFragments = undefined;
var groupId = undefined;
var tabId = undefined;
var groups = [];
var tabData = undefined;
var groupData = undefined;
window.summaryInterval = undefined;
var userData = {};

var callback = function() {
  fetch(API_URL + "/status/" + uniqueId + `?api_version=${API_VERSION}`, { method: "GET" })
    .then(response => response.json())
    .then(data => getGroupsData(data))
    .then(data => {
      groups = typeof data == "object" ? data : [];
      groupData = groups.filter(group => {
        return group.id == groupId;
      })[0];
      tabData = groupData.tabs.filter(tab => {
        return tab.tabId == tabId;
      })[0];
      initPage();
    })
    .catch(() => {
      new CustomModal(VALIDATION_MESSAGES.serverError, "", "");
    });
};

export function initTabPage() {
  userData = getUserData();
  uniqueId = userData["userId"];
  urlFragments = window.location.href.split("?")[1].split("&");
  groupId = parseInt(
    urlFragments
      .filter(fragment => {
        return fragment.indexOf("t=") > -1;
      })[0]
      .replace("t=", "")
  );
  tabId = urlFragments
    .filter(fragment => {
      return fragment.indexOf("id=") > -1;
    })[0]
    .replace("id=", "");
  callback();
}

function initPage() {
  $(".main-header .title").text(tabData.title);
  $(".languages").text(tabData.language ? tabData.language.toUpperCase() : "");
  var urlParams = new URLSearchParams(window.location.search);
  var redirectParam = urlParams.get("redirect");
  var redirectPage = redirectParam
    ? redirectParam == 1
      ? "favourite.html"
      : `search.html`
    : `topic.html?id=${groupId}`;
  $("#back-to-topics-btn")
    .closest("a")
    .attr("href", redirectPage);
  $(".main-header .group-color").css({ "background-color": groupData.color });
  $(".reading-time").text(tabData.readTime ? formatMinutes(tabData.readTime) : "N/A");
  initFactsAndSummary();
  initBookmarkBtn();
  initShareButton();
  initSimilarTabs();
  initFavouriteButton();
  initActionsMenu();
  $("#close-tabs-btn").on("click", function() {
    var tabCloseData = { id: tabData.tabId, url: tabData.url };
    closeTabsFromButton([tabCloseData], uniqueId, () => {
      navigateToPage(`topic.html?id=${groupId}`);
    });
  });
  $("#open-link-btn").on("click", function() {
    openTabOrUrl(tabData.tabId, tabData.url);
  });
  $("main").tooltip();
  updateFavouriteTabButton(tabData["isFavourite"]);
  TourMessageManager.initByPage();
}

function initBookmarkBtn() {
  if (getBrowser() == "safari" || isMobile() || isWebVersion()) {
    $(".bookmark-btn").remove();
    $("#actions-dropdown-container").addClass("one-less-option");
    return;
  }
  $(".bookmark-btn").on("click", function() {
    bookmarkTabs([{ url: tabData.url, title: tabData.title }], groupData.name);
  });
}

function initShareButton() {
  $(".share-btn").on("click", function() {
    toggleShareLinkContainer(tabData.url);
  });
}

function updateFavouriteTabButton(isFavourite) {
  var button = $(".favourite-btn");
  var imgSrc = "";
  if (isFavourite) {
    button.addClass("favourite");
    imgSrc = "_remove";
    button.find(".action-text").text("Remove from favourite tabs");
  } else {
    button.removeClass("favourite");
    button.find(".action-text").text("Add to favourite tabs");
  }
  button.find("img").attr("src", `images/favourite${imgSrc}.svg`);
}

function initFavouriteButton() {
  $(".favourite-btn").on("click", function() {
    var wasFavourite = $(this).hasClass("favourite");
    fetch(API_URL + "/favourite/" + uniqueId + `?api_version=${API_VERSION}`, {
      method: "POST",
      headers: HEADERS,
      body: JSON.stringify({ tabId })
    })
      .then(response => {
        if (response.ok) {
          var message = wasFavourite ? "tabFavouriteRemove" : "tabFavourite";
          updateFavouriteTabButton(!wasFavourite);
          addPopupMessage(message);
        }
      })
      .catch(() => {
        addPopupMessage("tabFavouriteError");
      });
  });
}

function initSimilarTabs() {
  fetch(API_URL + "/tab_similarity/" + uniqueId + `?api_version=${API_VERSION}`, {
    method: "POST",
    headers: HEADERS,
    body: JSON.stringify({ tabId: tabId, clusterId: groupId })
  })
    .then(function(response) {
      if (!response.ok) {
        throw new Error("The similar tabs can't be generated right now. Please try again later.");
      }
      return response.json();
    })
    .then(function(data) {
      if (data.status === "success") {
        var similarTabs = data.similarity || {};
        similarTabs = groupData.tabs.filter(function(tab) {
          return similarTabs.includes(tab.tabId);
        });

        const similarTabsContainer = $("#similar-tabs-container");
        similarTabsContainer.empty();

        if (similarTabs.length === 0) {
          similarTabsContainer.append('<span class="no-data invisible transition">No similar tabs found.</span>');
          setTimeout(() => {
            similarTabsContainer.find(".no-data").removeClass("invisible");
          }, 300);
        }

        var listItemsContents = [];
        similarTabs.forEach(function(similarTab) {
          const domain = similarTab.url.match(/^(?:https?:\/\/)?([^/]+)/)[1];
          const tabLink = `tab.html?id=${similarTab.tabId}&t=${groupId}`;
          var title = similarTab.title ? `${escapeHTML(similarTab.title)} - ` : "";
          listItemsContents.push(`<a href="${tabLink}">${title}${title ? domain : similarTab.url}</a>`);
        });
        addCustomOrderedList(listItemsContents, similarTabsContainer, ["similar-tab"]);
      } else {
        $("#similar-tabs-container").append(
          "<span>The similar tabs can't be generated right now. Please try again later.</span>"
        );
      }
    })
    .catch(function(err) {
      $("#similar-tabs-container").append(`<span>${err.message}</span>`);
    });
}

function initFactsAndSummary() {
  var fetchFactsAndSummary = function() {
    fetch(API_URL + "/summary/" + uniqueId + `?api_version=${API_VERSION}`, {
      method: "POST",
      headers: HEADERS,
      body: JSON.stringify({ tabId: tabData.id })
    })
      .then(function(response) {
        if (!response.ok) {
          throw new Error();
        }
        response.json().then(function(data) {
          if (data.status && data.status == "generated" && data.summary) {
            clearInterval(summaryInterval);
            $("#tab-main .summary-container").text(data.summary);
            if (typeof data.facts == "string") {
              try {
                var facts = JSON.parse(data.facts);
                data.facts = facts;
              } catch (error) {
                data.facts = "No stats for this page.";
              }
            }
            if (typeof data.facts == "object") {
              if (!data.facts.length) {
                $("#facts-placeholder-info").text("No stats for this page.");
              } else {
                $("#facts-placeholder-info").remove();
              }
              for (var fact of data.facts) {
                $("#topic-facts").append(`<li><span class="li-text">${fact}</span></li>`);
              }
            } else {
              $("#facts-placeholder-info").text("No stats for this page.");
              $("#facts-placeholder-info").removeClass("hidden");
              $("#facts-placeholder-info").removeClass("transparent");
            }
            if (data.summary.indexOf("TabCrunch was unable to analyse and categorise") !== -1) {
              $("#retry-btn").removeClass("hidden");
              $("#retry-btn").removeClass("transparent");
              $("#retry-btn").on("click", function() {
                fetch(API_URL + "/summary/" + uniqueId + `?api_version=${API_VERSION}`, {
                  method: "POST",
                  headers: HEADERS,
                  body: JSON.stringify({ tabId: tabData.id, retry: 1 })
                })
                  .then(function(response) {
                    if (!response.ok) {
                      throw new Error();
                    }
                  })
                  .then(function() {
                    window.location.reload();
                  });
              });
            }
          }
          if (data.status && !data.summary) {
            $("#facts-placeholder-info").text("Generating data...");
            $("#tab-main .summary-container").text("Generating summary...");
            $("#facts-placeholder-info").text("Generating data...");
            $("#facts-placeholder-info").removeClass("hidden");
            $("#facts-placeholder-info").removeClass("transparent");
          }
          $(".transition-container").removeClass("transparent");
          $("#tab-main").removeClass("transparent");
        });
      })
      .catch(() => {
        $("#tab-main .summary-container").text("The tab summary can't be generated right now. Please try again later.");
        $("#facts-placeholder-info").text("The key points can't be generated right now. Please try again later.");
        $("#facts-placeholder-info").removeClass("hidden");
        $("#facts-placeholder-info").removeClass("transparent");
        $("#tab-main").removeClass("transparent");
      });
  };
  summaryInterval = setInterval(fetchFactsAndSummary, 10000);
  fetchFactsAndSummary();
}
