import { isMobile, isWebVersion } from "./common.js";
import { getDeviceName, getUserData, setUserData } from "./panel.js";

export class TourMessageManager {
  static stepsIndexes = [
    "welcome",
    "menu",
    "analysis",
    "run",
    "closeTabs",
    "seeAnalysis",
    "closeTabsFromGroups",
    "groups",
    "editTitle",
    "share",
    "summary",
    "favourite",
    "history",
    "historyBubble",
    "historyClosed",
    "historyNew",
    "importMobile",
    "import",
    "finish"
  ];
  static titles = {
    welcome: "Welcome to TabCrunch! Fancy a tour?",
    menu: "What's on the menu?",
    analysis: "Run tab analysis from here",
    run: "Do you want to analyse all open tabs or just some?",
    closeTabs: "With your tabs analysed, you can free your browser of clutter.",
    seeAnalysis: "See analysis",
    groups: "All your tab groups in one place",
    editTitle: "Edit the title of your tab group",
    share: "Share tab groups",
    summary: "Tab summary",
    favourite: "Add tabs to favourites",
    history: "Let's check out your tab analysis history",
    historyBubble: "A chronological timeline of all past tab analysis",
    historyClosed: "Closed tabs since your last analysis",
    historyNew: "All new tabs since your last analysis",
    import: "Import tabs straight into TabCrunch",
    importMobile: "Import your mobile tabs",
    finish: ""
  };
  static texts = {
    welcome: "TabCrunch helps you analyse, group, and summarise your tabs...and much more. Let's show you around.",
    menu: `From here you can:
          <ul>
              <li>Run new tab analysis</li>
              <li>Access your tab summaries and stats</li>
              <li>Check your tab history</li>
              <li>Find the tab you need with a quick search</li>
              <li>Import and export tabs and tab groups</li>
              <li>and more...</li>
          </ul>
          <span id="menu-tour-text" class="next-step-text">Click on <span class="gradient-btn"><img src="images/open-menu.svg"></span> to proceed.</span>`,
    analysis: `TabCrunch will use a Large Language Model to analyse your tabs' content. It will then group your tabs and generate bite-size summaries of each tab and tab group.
      <span class="next-step-text">Click on "Analyse tabs" to proceed.</span>`,
    run: `
      <p>
        <span class="text-black">Tap here</span> to see a list of all tabs currently open in your browser. Remove any you do not wish to analyse. 
        If you wish to analyse all open tabs, simply <span class="text-black">click on “Run analysis”</span>.
      </p>
      <p>TabCrunch tries to avoid analysing tabs containing medical, financial, and other types of sensitive information by default.</p>
      <span id="menu-tour-text" class="next-step-text">Choose either option to proceed.</span>`,
    closeTabs: `Save up memory and reduce load on your device by closing all analysed tabs in the browser. Don't worry, those tabs are safely stored in TabCrunch and you can easily open them again in your browser at any point.`,
    seeAnalysis:
      "Tap here to see your tab groups. You can also access them at any time through the Tab groups section in the menu.",
    closeTabsFromGroups: `Save up memory and reduce load on your device by closing all analysed tabs in the browser. Don't worry, those tabs are safely stored in TabCrunch and you can easily open them again in your browser at any point.`,
    groups:
      "See summaries of each tab group along with the number of tabs, reading time, and languages per tab group. Click on any tab group to see the tabs and access the tab summaries.",
    editTitle: "Group titles are generated automatically, but your can manually change them as you wish.",
    share: "You can share an entire tab group and collaborate with others.",
    summary: "Tap here to see a summary and analytics for each tab.",
    favourite:
      "You can access all your favourite tabs from one place — the Favourite tabs section, accessible from the menu.",
    history:
      "Here you can see details of each tab analysis you have run in the past. TabCrunch always keeps your history, so you can safely close your tabs in the browser and free up your device.",
    historyBubble: "You can see when each analysis was ran along with a list of all analysed tabs.",
    historyClosed: "Click on the icon at the bottom to see a full list of all closed tabs.",
    historyNew:
      "Whenever you run a new analysis, TabCrunch analyses all currently open tabs. This means that we process once again all previously analysed tabs that are still open as well as the newly open tabs. Here, you can see a full list of all the new tabs you've opened since your last analysis.",
    import:
      "No need to have all tabs open in your browser. If you have a list of tabs in a CSV, TXT, or HTML file, you can simply upload them here.",
    importMobile:
      "<p>Analyse tabs across devices. You can simply copy all your tabs from your mobile browser and paste them here. You can then run analysis on all your mobile and desktop tabs.</p><p>Alternatively, you can install the TabCrunch extension on your mobile.</p>",
    finish: `
        <img src="images/tour_finish.png">
        <div id="finish-message">
          <span class="bigger-font">That's it, you've reached the end!</span><span>We hope this helped.</span>
        </div>`
  };
  static buttonTexts = {
    closeTabs: "OK, next",
    closeTabsFromGroups: "OK, next",
    editTitle: "OK, next",
    share: "OK, next",
    favourite: "OK, next",
    history: "OK, let's take a look",
    historyBubble: "OK, next",
    historyClosed: "OK, next",
    historyNew: "OK, next",
    import: "OK, got it",
    importMobile: "Got it"
  };
  static currentStep = "";

  static initByStep(validSteps, newStep) {
    var userData = getUserData();
    var tourStep = userData.tourStep;

    if (!TourMessageManager.isActiveTour()) {
      return;
    }

    if (validSteps.indexOf(tourStep) != -1) {
      TourMessageManager.updateStep(newStep);
      TourMessageManager.updateMessage(newStep);
      return true;
    } else {
      return false;
    }
  }

  static initByPage(newStep = null) {
    var page = window.location.pathname;
    var userData = getUserData();
    var tourStep = userData.tourStep || "welcome";
    var tourValidation = () => true;
    var validSteps = [];

    if (!TourMessageManager.isActiveTour()) {
      return;
    }

    switch (page) {
      case "/analyse.html":
        validSteps = ["run", "finish"];
        tourValidation = () => !$(".popup-wrapper").length;
        break;
      case "/tab.html":
        validSteps = ["favourite"];
        break;
      case "/topic.html":
        validSteps = ["editTitle", "share", "summary"];
        break;
      case "/topics.html": {
        validSteps = [
          "welcome",
          "groups",
          "closeTabs",
          "closeTabsFromGroups",
          "finish",
          "menu",
          "analysis",
          "seeAnalysis"
        ];
        if (!tourStep || tourStep == "menu" || tourStep == "analysis") {
          newStep = "welcome";
        }
        if ((tourStep == "closeTabs" || tourStep == "groups") && !$(".grid-columns .grid-element").length) {
          return;
        }
        if (tourStep == "closeTabs") {
          newStep = "closeTabsFromGroups";
        } else if (tourStep == "seeAnalysis") {
          newStep = "groups";
        }
        break;
      }
      case "/history.html":
        validSteps = ["history", "historyBubble", "historyClosed", "historyNew"];
        break;
      case "/import.html":
        validSteps = ["import", "importMobile"];
        break;
    }

    if (validSteps.indexOf(tourStep) != -1 && tourValidation()) {
      if (!newStep) {
        newStep = tourStep;
        TourMessageManager.currentStep = tourStep;
      } else {
        TourMessageManager.updateStep(newStep);
      }
      TourMessageManager.updateMessage(newStep);
      return true;
    } else {
      return false;
    }
  }

  static renderMessage(additionalAttributes = {}) {
    var currentSrtepIndex = this.stepsIndexes.indexOf(this.currentStep);
    var currentStep = this.currentStep;
    var nextButtonText = this.buttonTexts[currentStep];
    var nextButtonHTML = nextButtonText
      ? `<button id="next-step-btn" class="gradient-btn">${nextButtonText}</button>`
      : "";
    additionalAttributes["customId"] = "tour-modal";
    additionalAttributes["disableDefaultClose"] = true;

    if (currentStep == "welcome") {
      var messageFooter = `<button id="skip-tour-btn" class="border-btn">Remind me next time</button>
                <button id="next-step-btn" class="gradient-btn">Let's go</button>`;
    } else {
      var messageFooter =
        currentStep != "finish"
          ? `<div id="step-counter">${currentSrtepIndex + 1} of ${this.stepsIndexes.length}</div>${nextButtonHTML}`
          : "";
    }

    new CustomModal(
      `<span id="tour-text-container">${this.texts[currentStep]}</span>`,
      this.titles[currentStep],
      messageFooter,
      additionalAttributes
    );

    var style = this.getStyleForStep();
    $(".modal-content-holder").css(style);

    $(".custom-modal #close-modal-icon").on("click", () => this.closeCallback());

    $("#next-step-btn").on("click", () => {
      this.updateStep();
      this.autoButtonAction();
    });
  }

  static updateStep(selectedStep = "") {
    if (!TourMessageManager.isActiveTour()) {
      return;
    }

    var stepIndex = this.currentStep ? this.stepsIndexes.indexOf(this.currentStep) + 1 : 0;
    this.currentStep = selectedStep ? selectedStep : this.stepsIndexes[stepIndex];

    if (this.currentStep == "historyClosed") {
      var sidePanelElement = $("#history-tabs .left-node:not(.invisible):first");
      if (!sidePanelElement.length) {
        this.currentStep = this.stepsIndexes[++stepIndex];
      }
    }
    if (this.currentStep == "historyNew") {
      var sidePanelElement = $("#history-tabs .right-node:not(.invisible):first");
      if (!sidePanelElement.length) {
        this.currentStep = this.stepsIndexes[++stepIndex];
      }
    }

    CustomModal.closeModal("#tour-modal .modal-content-holder");
    $(".tour-highlight").removeClass("tour-highlight");
    setUserData({ tourStep: this.currentStep });
  }

  static autoButtonAction() {
    switch (this.currentStep) {
      case "menu":
        this.updateMessage();
        break;
      case "analysis":
        $("#open-menu-btn-conainer").trigger("click");
        break;
      case "run":
        $("#menu-options-container #tab-analyse").trigger("click");
        break;
      case "groups":
        if (window.location.href.indexOf("topics.html") == -1) {
          navigateToPage("topics.html");
        } else {
          this.updateMessage(this.currentStep);
        }
        break;
      case "editTitle":
        $("#grid-holder .grid-element:first-of-type").click();
        break;
      case "share":
        setTimeout(() => {
          this.updateMessage(this.currentStep);
        }, 400);
        break;
      case "summary":
      case "historyBubble":
      case "historyClosed":
      case "historyNew":
      case "import":
      case "seeAnalysis":
        this.updateMessage(this.currentStep);
        break;
      case "favourite":
        $(".table tbody tr:first-of-type td:nth(4) a").click();
        break;
      case "history":
        navigateToPage("history.html");
        break;
      case "importMobile":
        navigateToPage("import.html?p=import");
        break;
      case "finish":
        navigateToPage("topics.html");
      default:
        break;
    }
  }

  static updateMessage() {
    if (!TourMessageManager.isActiveTour()) {
      return;
    }

    var additionalAttributes = {};
    var highlightedElement = undefined;
    var currentStep = this.currentStep;
    var timeoutTime = 500;
    if (["historyClosed", "historyNew"].indexOf(currentStep) != -1) {
      timeoutTime = 1200;
    } else if (["share", "summary"].indexOf(currentStep) != -1) {
      timeoutTime = 800;
    } else if (currentStep == "favourite") {
      timeoutTime = 2000;
    }

    if (currentStep == "share" || (currentStep == "summary" && $("#actions-dropdown-container").hasClass("active"))) {
      $(".action-menu-btn").trigger("click");
    } else if (currentStep == "favourite") {
      setTimeout(() => {
        $(".action-menu-btn").trigger("click");
      }, 1200);
    }

    setTimeout(() => {
      switch (currentStep) {
        case "menu":
          additionalAttributes = { additionalClasses: "wider" };
          highlightedElement = "#main-menu-container";
          break;
        case "analysis":
          additionalAttributes = { hideBackground: true };
          highlightedElement = "#expanded-menu-container";
          break;
        case "run":
          additionalAttributes = { hideBackground: true, additionalClasses: "light-modal wider extended-pointer" };
          highlightedElement = "#view-tabs-list, #graphic-container";
          break;
        case "closeTabs":
          additionalAttributes = { hideBackground: true, additionalClasses: "light-modal wider extended-pointer" };
          highlightedElement = "#analyse-reults, #graphic-container";
          break;
        case "seeAnalysis":
          additionalAttributes = { hideBackground: true, additionalClasses: "light-modal" };
          highlightedElement = "#graphic-container";
          break;
        case "closeTabsFromGroups":
          highlightedElement = "#close-groups";
          break;
        case "groups":
          additionalAttributes = { additionalClasses: "point-downward wider" };
          break;
        case "editTitle":
          highlightedElement = ".edit-btn";
          break;
        case "share":
          var positionClass =
            $(".share-btn").offset().left < $("body").outerWidth() / 2 ? "extended-pointer" : "point-right";
          additionalAttributes = { additionalClasses: `wider ${positionClass}` };
          break;
        case "summary":
          additionalAttributes = { additionalClasses: "point-downward wider" };
          highlightedElement = "table tr td.summary";
          break;
        case "favourite":
          var positionClass = $("body").outerWidth() / 2 < $(".favourite-btn").offset().left ? "point-right" : "";
          additionalAttributes = { additionalClasses: `wider ${positionClass}` };
          break;
        case "history":
          break;
        case "historyBubble":
          highlightedElement = "#history-tabs .graphic-container:first";
          break;
        case "historyClosed": {
          highlightedElement = "#history-tabs .left-node:not(.invisible):first";
          if (document.querySelector("#history-tabs .left-node:not(.invisible)")) {
            document.querySelector("#history-tabs .left-node:not(.invisible)").scrollIntoView({ behavior: "smooth" });
          }
          break;
        }
        case "historyNew":
          additionalAttributes = { additionalClasses: "point-right" };
          highlightedElement = "#history-tabs .right-node:not(.invisible):first";
          if (document.querySelector("#history-tabs .right-node:not(.invisible)")) {
            document.querySelector("#history-tabs .right-node:not(.invisible)").scrollIntoView({ behavior: "smooth" });
          }
          break;
        case "import":
          highlightedElement = "#import-file-container";
          break;
        case "importMobile":
          highlightedElement = "#import-links-container";
          break;
        case "finish":
          additionalAttributes = { additionalClasses: "image-background" };
        default:
          break;
      }
    }, 350);

    setTimeout(() => {
      if (currentStep != "groups") {
        this.renderMessage(additionalAttributes);
      }
      switch (currentStep) {
        case "welcome":
          $(".modal-content-holder").addClass("no-pointer");
          $("#skip-tour-btn").on("click", () => {
            setUserData({ lastTourReminder: new Date().getTime(), currentStep: "" });
            CustomModal.closeModal("#tour-modal .modal-content-holder");
          });
          break;
        case "menu":
          $(".modal-content-holder").removeClass("no-pointer");
          break;
        case "analysis":
          $("#expanded-menu-container").css({ width: "auto" });
          break;
        case "groups":
          var gridElements = $("#grid-holder .grid-element");
          if (gridElements.length) {
            this.renderMessage(additionalAttributes);
            highlightedElement = ".grid-element";
          }
          break;
        case "history":
          $(".modal-content-holder").addClass("no-pointer");
          break;
        case "finish":
          $(".modal-content-holder").addClass("no-pointer");
          setUserData({ takingTour: 0, tourFinished: 1 });
        default:
          break;
      }
      if (highlightedElement) {
        $(highlightedElement).addClass("tour-highlight");
      }
    }, timeoutTime);
  }

  static getStyleForStep() {
    var currentStep = this.currentStep;
    if (currentStep == "welcome") {
      return {
        left: "50%",
        top: "50%",
        translate: "-50% -50%"
      };
    } else if (currentStep == "menu") {
      return {
        left: "75px",
        top: "50%",
        translate: "0 -50%"
      };
    } else if (currentStep == "analysis") {
      var analyseMenuButton = $("#menu-options-container .menu-option:nth-of-type(1)");
      var horizontalOffset =
        analyseMenuButton.offset().left +
        analyseMenuButton.find(".icon-img").outerWidth() +
        analyseMenuButton.find("#tab-analyse").outerWidth() +
        45;
      return {
        top: analyseMenuButton.offset().top + analyseMenuButton.outerHeight() / 2,
        left: `${horizontalOffset}px`,
        translate: "0 -50%"
      };
    } else if (currentStep == "run") {
      var seeTabsButton = $("#view-tabs-list");
      var graphicContainer = $("#graphic-container");
      var horizontalOffset = 75 + seeTabsButton.outerWidth() / 2;
      var verticalOffset = graphicContainer.outerHeight() / 2 + seeTabsButton.outerHeight() / 2;
      return {
        top: `calc(50% - ${verticalOffset}px)`,
        left: "50%",
        translate: `${horizontalOffset}px -50%`,
        "max-width": "460px",
        "max-height": "360px"
      };
    } else if (currentStep == "closeTabs") {
      var highlightedElement = $("#close-tabs-text-button");
      var left = "50%";
      var horizontalOffset = 75 + highlightedElement.outerWidth() / 2;
      var verticalOffset = highlightedElement.offset().top + highlightedElement.outerHeight() / 2;
      return {
        top: `${verticalOffset}px`,
        left,
        translate: `${horizontalOffset}px -50%`
      };
    } else if (currentStep == "closeTabsFromGroups") {
      var highlightedElement = $("#close-groups");
      var left = highlightedElement.offset().left + highlightedElement.outerWidth() + 25;
      var verticalOffset = highlightedElement.offset().top + highlightedElement.outerHeight() / 2;
      return {
        top: `${verticalOffset}px`,
        left,
        translate: `0 -50%`
      };
    } else if (currentStep == "seeAnalysis") {
      var highlightedElement = $("#graphic-container");
      var left = "50%";
      var horizontalOffset = 20 + highlightedElement.outerWidth() / 2;
      var verticalOffset = highlightedElement.offset().top + highlightedElement.outerHeight() / 2;
      return {
        top: `${verticalOffset}px`,
        left,
        translate: `${horizontalOffset}px -50%`
      };
    } else if (currentStep == "groups") {
      var verticalOffset = $("#grid-holder .grid-element:first-of-type").offset().top - 20;
      return {
        left: "auto",
        top: verticalOffset,
        right: "20px",
        translate: "0 -100%"
      };
    } else if (currentStep == "editTitle") {
      var horizontalOffset = $(".edit-btn").offset().left + 60;
      return {
        top: "110px",
        left: `${horizontalOffset}px`,
        right: "auto",
        translate: "0 -50%",
        "max-width": "400px"
      };
    } else if (currentStep == "share") {
      var menuIsOnLeftSide = $(".share-btn").offset().left < $("body").outerWidth() / 2;
      var horizontalOffset = menuIsOnLeftSide
        ? $(".share-btn").offset().left + $(".share-btn").outerWidth() / 2 + 70
        : $(".share-btn").offset().left - 30;
      var verticalOffset = $(".share-btn").offset().top + $(".share-btn").outerHeight() / 2;
      return {
        top: `${verticalOffset}px`,
        left: `${horizontalOffset}px`,
        right: "145px",
        translate: menuIsOnLeftSide ? "0 -50%" : "-100% -50%"
      };
    } else if (currentStep == "summary") {
      var summaryButton = $(".table tbody tr:first-of-type .summary");
      var horizontalOffset = summaryButton.offset().left + summaryButton.outerWidth() / 2;
      var verticalOffset = summaryButton.offset().top - 20;
      return {
        left: `${horizontalOffset}px`,
        top: `${verticalOffset}px`,
        bottom: "auto",
        right: "auto",
        translate: "-50% -100%",
        "max-width": "400px"
      };
    } else if (currentStep == "favourite") {
      var menuIsOnLeftSide = $(".favourite-btn").offset().left < $("body").outerWidth() / 2;
      var horizontalOffset = menuIsOnLeftSide
        ? $(".favourite-btn").offset().left + $(".favourite-btn").outerWidth()
        : $(".favourite-btn").offset().left - 30;
      var verticalOffset = $(".favourite-btn").offset().top + $(".favourite-btn").outerHeight() / 2;
      return {
        left: `${horizontalOffset}px`,
        top: `${verticalOffset}px`,
        right: "145px",
        translate: menuIsOnLeftSide ? "0 -50%" : "-100% -50%"
      };
    } else if (currentStep == "history") {
      return {
        left: "auto",
        top: "40px",
        right: "40px",
        translate: "0 0"
      };
    } else if (currentStep == "historyBubble") {
      var bubbleElement = $("#history-tabs .graphic-container:first");
      var verticalOffset = bubbleElement.offset().top + bubbleElement.outerHeight() / 2;
      var horizontalOffset = 25 + bubbleElement.outerWidth() / 2;
      return {
        left: "50%",
        top: `${verticalOffset}px`,
        right: "auto",
        translate: `${horizontalOffset}px -50%`
      };
    } else if (currentStep == "historyClosed") {
      var sidePanelElement = $("#history-tabs .left-node:not(.invisible):first");
      var verticalOffset = sidePanelElement.offset().top + sidePanelElement.outerHeight() / 2;
      var horizontalOffset =
        sidePanelElement.outerWidth() + 25 + ($("main").outerWidth() - $("#history-tabs").outerWidth()) / 2;
      return {
        left: `${horizontalOffset}px`,
        top: `${verticalOffset}px`,
        translate: `0 -50%`
      };
    } else if (currentStep == "historyNew") {
      var sidePanelElement = $("#history-tabs .right-node:not(.invisible):first");
      var bubbleElement = $("#history-tabs .graphic-container:first");
      var verticalOffset = sidePanelElement.offset().top + sidePanelElement.outerHeight() / 2;
      var horizontalOffset =
        sidePanelElement.outerWidth() + 25 + ($("main").outerWidth() - $("#history-tabs").outerWidth()) / 2;
      return {
        left: "auto",
        right: `${horizontalOffset}px`,
        top: `${verticalOffset}px`,
        translate: `0 -50%`
      };
    } else if (currentStep == "import") {
      var highlightedElement = $("#import-file-container");
      var verticalOffset = highlightedElement.offset().top + highlightedElement.outerHeight() / 2;
      var horizontalOffset = highlightedElement.outerWidth() / 2 + 25;
      return {
        left: "50%",
        right: `auto`,
        top: `${verticalOffset}px`,
        translate: `${horizontalOffset}px -50%`
      };
    } else if (currentStep == "importMobile") {
      var highlightedElement = $("#import-links-container");
      var verticalOffset = highlightedElement.offset().top + highlightedElement.outerHeight() / 2;
      var horizontalOffset = highlightedElement.outerWidth() / 2 + 25;
      return {
        left: "50%",
        right: `auto`,
        top: `${verticalOffset}px`,
        translate: `${horizontalOffset}px -50%`
      };
    } else if (currentStep == "finish") {
      return {
        "max-width": "680px",
        width: "680px",
        "max-height": "100%",
        left: "50%",
        top: "50%",
        translate: "-50% -50%"
      };
    }
  }

  static closeCallback() {
    if (this.currentStep == "finish") {
      CustomModal.closeModal("#tour-text-container");
      return;
    }

    $("#tour-text-container").html(
      `Are you sure you'd like to skip the guided tour and never see this message again.<br> Alternatively, you can close it for now and we will remind you again next time. You will continue from where you left off.`
    );
    $("#tour-modal .modal-main-title").text("Skip the guided tour?");
    $("#tour-modal .modal-main-footer").html(
      `<button id="quit-tour-btn" class="border-btn">Never show this again</button>
      <button id="close-tour-btn" class="gradient-btn">Remind me next time</button>`
    );

    $("#tour-modal .close-modal").remove();
    $("#tour-modal").addClass("wider");
    $(".modal-content-holder").addClass("no-pointer");
    $(".tour-highlight").removeClass("tour-highlight");

    $("#close-tour-btn").on("click", function() {
      setUserData({ lastTourReminder: new Date().getTime(), takingTour: 0, tourFinished: 0 });
      CustomModal.closeModal("#tour-text-container");
    });
    $("#quit-tour-btn").on("click", function() {
      setUserData({ takingTour: 0, tourFinished: 1 });
      CustomModal.closeModal("#tour-text-container");
    });
  }

  static isActiveTour() {
    var userData = getUserData();
    var lastReminder = userData.lastTourReminder ? parseInt(userData.lastTourReminder) : 0;
    return (
      userData.takingTour == "1" &&
      new Date().getTime() - lastReminder >= 8 * 3600000 &&
      !userData.userId.includes("temp") &&
      getDeviceName() &&
      !isMobile() &&
      !isWebVersion()
    );
  }
}
